<?php
declare(strict_types=1);

final class OrdersController {
  public static function list(string $incidentId): void {
    $u = require_login();
    $pdo = db();
    if ($u['role'] !== 'admin') {
      $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
      $chk->execute([':iid'=>$incidentId, ':oid'=>$u['org_id']]);
      if (!$chk->fetchColumn()) api_error(403, 'not a participant');
    }

    $stmt = $pdo->prepare("SELECT o.id, o.incident_id, o.created_by_user_id, o.template_key, o.title, o.status,
        o.assigned_team_id, o.assigned_org_id, o.priority, o.fields_json,
        o.created_at, o.accepted_at, o.started_at, o.completed_at,
        t.name AS team_name, org.name AS assigned_org_name,
        tr.id AS transfer_id, tr.state AS transfer_state, tr.mode AS transfer_mode,
        tr.from_org_id AS transfer_from_org_id, tr.to_org_id AS transfer_to_org_id,
        tr.created_at AS transfer_created_at, tr.decided_at AS transfer_decided_at,
        org_from.name AS transfer_from_org_name, org_to.name AS transfer_to_org_name
      FROM \"order\" o
      LEFT JOIN team t ON t.id=o.assigned_team_id
      LEFT JOIN org ON org.id=o.assigned_org_id
      LEFT JOIN LATERAL (
        SELECT tt.*
        FROM transfer tt
        WHERE tt.type='order_transfer'
          AND tt.incident_id=o.incident_id
          AND (tt.payload_json->>'order_id') = o.id::text
        ORDER BY tt.created_at DESC
        LIMIT 1
      ) tr ON TRUE
      LEFT JOIN org org_from ON org_from.id = tr.from_org_id
      LEFT JOIN org org_to ON org_to.id = tr.to_org_id
      WHERE o.incident_id=:iid
      ORDER BY o.created_at DESC");
    $stmt->execute([':iid'=>$incidentId]);
    $orders = $stmt->fetchAll();
    foreach ($orders as &$o) {
      $o['fields'] = json_decode($o['fields_json'] ?? '{}', true);
      unset($o['fields_json']);

      // Transfer info (latest) – used by the UI to show what happened / where the order went.
      if (!empty($o['transfer_id'])) {
        $o['transfer'] = [
          'id'=>$o['transfer_id'],
          'state'=>$o['transfer_state'],
          'mode'=>$o['transfer_mode'],
          'from_org_id'=>$o['transfer_from_org_id'],
          'to_org_id'=>$o['transfer_to_org_id'],
          'from_org_name'=>$o['transfer_from_org_name'],
          'to_org_name'=>$o['transfer_to_org_name'],
          'created_at'=>$o['transfer_created_at'],
          'decided_at'=>$o['transfer_decided_at'],
        ];
      }
      unset(
        $o['transfer_id'], $o['transfer_state'], $o['transfer_mode'],
        $o['transfer_from_org_id'], $o['transfer_to_org_id'],
        $o['transfer_created_at'], $o['transfer_decided_at'],
        $o['transfer_from_org_name'], $o['transfer_to_org_name']
      );
    }
    api_json(['ok'=>true, 'orders'=>$orders]);
  }

  public static function create(string $incidentId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $title = trim((string)($in['title'] ?? ''));

    $templateKey = trim((string)($in['template_key'] ?? ''));
    $priority = (int)($in['priority'] ?? 3);
    $fields = $in['fields'] ?? [];
    if (!is_array($fields)) $fields = [];

    // If a template is selected, prefill/validate fields and default the title.
    if ($templateKey !== '') {
      $tpls = setting('orderTemplates', []);
      $tpl = null;
      foreach ($tpls as $t) {
        if (is_array($t) && (string)($t['key'] ?? '') === $templateKey) { $tpl = $t; break; }
      }
      if ($tpl) {
        if ($title === '') $title = (string)($tpl['label'] ?? $tpl['key'] ?? 'Auftrag');
        // Only allow keys defined in template (keeps payload small and predictable)
        $allowed = [];
        foreach (($tpl['fields'] ?? []) as $f) {
          if (is_array($f) && isset($f['key'])) $allowed[(string)$f['key']] = true;
        }
        if ($allowed) {
          $fields = array_filter($fields, fn($v, $k) => isset($allowed[(string)$k]), ARRAY_FILTER_USE_BOTH);
        }
      }
    }

    if ($title === '') api_error(400, 'title required');

    $assignedTeamId = (string)($in['assigned_team_id'] ?? '');
    $assignedOrgId = (string)($in['assigned_org_id'] ?? $u['org_id']);

    // GF can only assign to own org by default (EL/Admin can assign any participating org)
    if (!in_array($u['role'], ['admin','el'], true) && $assignedOrgId !== $u['org_id']) {
      api_error(403, 'cannot assign order to other org');
    }

    $stmt = db()->prepare('INSERT INTO "order"(id, incident_id, created_by_user_id, template_key, title, status, assigned_team_id, assigned_org_id, priority, fields_json)
      VALUES (gen_random_uuid(), :iid, :uid, :tk, :t, :st, NULLIF(:tid,\'\')::uuid, :oid, :p, :f::jsonb)
      RETURNING id');
    $stmt->execute([
      ':iid'=>$incidentId, ':uid'=>$u['id'], ':tk'=>$templateKey, ':t'=>$title, ':st'=>'created',
      ':tid'=>$assignedTeamId, ':oid'=>$assignedOrgId, ':p'=>$priority,
      ':f'=>json_encode($fields, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
    ]);
    $id = (string)$stmt->fetchColumn();

    db()->prepare('INSERT INTO audit_log(id, incident_id, actor_user_id, action, entity_type, entity_id, meta_json)
      VALUES (gen_random_uuid(), :iid, :uid, :act, :et, :eid, :m::jsonb)')
      ->execute([':iid'=>$incidentId, ':uid'=>$u['id'], ':act'=>'order.create', ':et'=>'order', ':eid'=>$id, ':m'=>json_encode(['title'=>$title,'assigned_org_id'=>$assignedOrgId])]);

    EventsController::emitIncident($incidentId, null, 'all', 'orders.updated', ['incident_id'=>$incidentId]);
    api_json(['ok'=>true, 'id'=>$id], 201);
  }

  public static function update(string $orderId): void {
    $u = require_role(['admin','el','gf']);
    $pdo = db();
    $stmt = $pdo->prepare('SELECT id, incident_id, assigned_org_id FROM "order" WHERE id=:id');
    $stmt->execute([':id'=>$orderId]);
    $o = $stmt->fetch();
    if (!$o) api_error(404, 'order not found');

    // GF limited: can only modify orders assigned to own org
    if (!in_array($u['role'], ['admin','el'], true) && $o['assigned_org_id'] !== $u['org_id']) api_error(403, 'forbidden');

    $in = json_input();
    $fields = [];
    $params = [':id'=>$orderId];

    if (isset($in['title'])) { $fields[]='title=:t'; $params[':t']=trim((string)$in['title']); }
    if (isset($in['status'])) {
      $fields[]='status=:s';
      $params[':s']=(string)$in['status'];
      // set timestamps on transitions
      if ($params[':s']==='accepted') $fields[]='accepted_at=COALESCE(accepted_at, now())';
      if ($params[':s']==='started')  $fields[]='started_at=COALESCE(started_at, now())';
      if ($params[':s']==='completed')$fields[]='completed_at=COALESCE(completed_at, now())';
    }
    if (isset($in['assigned_team_id'])) { $fields[]='assigned_team_id=NULLIF(:tid,\'\')::uuid'; $params[':tid']=(string)$in['assigned_team_id']; }
    if (isset($in['priority'])) { $fields[]='priority=:p'; $params[':p']=(int)$in['priority']; }
    if (isset($in['fields']) && is_array($in['fields'])) { $fields[]='fields_json=:f::jsonb'; $params[':f']=json_encode($in['fields'], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); }

    if (!$fields) api_error(400, 'nothing to update');

    $sql = 'UPDATE "order" SET ' . implode(',', $fields) . ', updated_at=now() WHERE id=:id';
    $pdo->prepare($sql)->execute($params);

    EventsController::emitIncident((string)$o['incident_id'], null, 'all', 'orders.updated', ['incident_id'=>$o['incident_id']]);
    api_json(['ok'=>true]);
  }

  public static function addNote(string $orderId): void {
    $u = require_role(['admin','el','gf']);
    $pdo = db();
    $stmt = $pdo->prepare('SELECT id, incident_id, assigned_org_id FROM "order" WHERE id=:id');
    $stmt->execute([':id'=>$orderId]);
    $o = $stmt->fetch();
    if (!$o) api_error(404, 'order not found');

    if (!in_array($u['role'], ['admin','el'], true) && $o['assigned_org_id'] !== $u['org_id']) api_error(403, 'forbidden');

    $in = json_input();
    $msg = trim((string)($in['message'] ?? ''));
    if ($msg === '') api_error(400, 'message required');

    $stmt = $pdo->prepare('INSERT INTO order_note(id, order_id, incident_id, author_user_id, message, created_at)
      VALUES (gen_random_uuid(), :oid, :iid, :uid, :m, now())');
    $stmt->execute([':oid'=>$orderId, ':iid'=>$o['incident_id'], ':uid'=>$u['id'], ':m'=>$msg]);

    EventsController::emitIncident((string)$o['incident_id'], null, 'all', 'orders.updated', ['incident_id'=>$o['incident_id']]);
    api_json(['ok'=>true]);
  }
}
